import React, { useRef, useState } from 'react';
import SignatureCanvas from 'react-signature-canvas';
import axios from 'axios';

import Config from './config';

// Función para recortar manualmente el canvas
const trimCanvas = (canvas) => {
    const ctx = canvas.getContext('2d');
    const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
    const data = imageData.data;
    
    let top = 0, bottom = canvas.height, left = 0, right = canvas.width;
    
    // Encontrar el borde superior
    topLoop: for (; top < bottom; top++) {
      for (let x = 0; x < right; x++) {
        const alpha = data[(top * canvas.width + x) * 4 + 3];
        if (alpha > 0) break topLoop;
      }
    }
    
    // Encontrar el borde inferior
    bottomLoop: for (; bottom > top; bottom--) {
      for (let x = 0; x < right; x++) {
        const alpha = data[((bottom - 1) * canvas.width + x) * 4 + 3];
        if (alpha > 0) break bottomLoop;
      }
    }
    
    // Encontrar el borde izquierdo
    leftLoop: for (; left < right; left++) {
      for (let y = top; y < bottom; y++) {
        const alpha = data[(y * canvas.width + left) * 4 + 3];
        if (alpha > 0) break leftLoop;
      }
    }
    
    // Encontrar el borde derecho
    rightLoop: for (; right > left; right--) {
      for (let y = top; y < bottom; y++) {
        const alpha = data[(y * canvas.width + (right - 1)) * 4 + 3];
        if (alpha > 0) break rightLoop;
      }
    }
    
    // Crear nuevo canvas con las dimensiones recortadas
    const trimmedCanvas = document.createElement('canvas');
    trimmedCanvas.width = right - left;
    trimmedCanvas.height = bottom - top;
    const trimmedCtx = trimmedCanvas.getContext('2d');
    
    trimmedCtx.drawImage(
      canvas,
      left, top, right - left, bottom - top,
      0, 0, right - left, bottom - top
    );
    
    return trimmedCanvas;
  };

const SignaturePad = () => {
  const sigCanvas = useRef(null);
  const [imageURL, setImageURL] = useState(null);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitMessage, setSubmitMessage] = useState('');

  // Limpiar el canvas de firma
  const clearSignature = () => {
    sigCanvas.current.clear();
    setImageURL(null);
    setSubmitMessage('');
  };

  // Guardar la firma como imagen PNG en Base64
  /*
  // Reduciendo el tamaño de la imagen
  const saveSignature = () => {
    if (sigCanvas.current.isEmpty()) {
      alert('Por favor, proporciona tu firma primero');
      return;
    }

    const canvas = sigCanvas.current.getCanvas();
    const trimmedCanvas = trimCanvas(canvas);
    const image = trimmedCanvas.toDataURL('image/png');
    
    setImageURL(image);
    setSubmitMessage('Firma guardada y recortada, lista para enviar');
  };
  */
  
  const saveSignature = () => {
    if (sigCanvas.current.isEmpty()) {
      alert('Por favor, proporciona tu firma primero');
      return;
    }
  
    // Enfoque alternativo
    const canvas = sigCanvas.current.getCanvas();
    const image = canvas.toDataURL('image/png');
    setImageURL(image);
    setSubmitMessage('Firma guardada, lista para enviar');
  };
  

  // Enviar la firma al servidor
  const submitSignature = async () => {
    if (!imageURL) {
      alert('Por favor, guarda tu firma primero');
      return;
    }

    setIsSubmitting(true);
    setSubmitMessage('Enviando firma...');

    try {
      // Extraer solo la parte Base64 de la URL de datos
      const base64Image = imageURL.split(',')[1];

      const url = Config.RUTA_API + "/saveFirma";
      const formData = new FormData();
        formData.append('signature', base64Image);
        formData.append('timestamp',  new Date().toISOString());

      const config = {
        headers: {
            "Content-Type": "application/x-www-form-urlencoded",
            "Authorization": Config.AUTHORIZATION
        },
      };

      // Reemplaza esta URL con tu endpoint real
      const response = await axios.post(
        url,
        formData,
        config
      );

      setSubmitMessage('Firma enviada con éxito!');
      console.log('Respuesta del servidor:', response.data);
    } catch (error) {
      console.error('Error al enviar la firma:', error);
      setSubmitMessage('Error al enviar la firma');
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div style={styles.container}>
      <h1>Captura de Firma</h1>
      
      <div style={styles.signatureContainer}>
        <SignatureCanvas
          ref={sigCanvas}
          canvasProps={{
            width: 500,
            height: 200,
            className: 'signature-canvas',
            style: styles.canvas
          }}
          penColor="black"
        />
      </div>

      <div style={styles.buttonGroup}>
        <button style={styles.button} onClick={clearSignature}>
          Limpiar Firma
        </button>
        <button style={styles.button} onClick={saveSignature}>
          Guardar Firma
        </button>
        <button 
          style={styles.button} 
          onClick={submitSignature}
          disabled={isSubmitting || !imageURL}
        >
          {isSubmitting ? 'Enviando...' : 'Enviar Firma'}
        </button>
      </div>

      {submitMessage && (
        <p style={submitMessage.includes('éxito') ? styles.success : styles.message}>
          {submitMessage}
        </p>
      )}

      {imageURL && (
        <div style={styles.preview}>
          <h3>Previsualización de la firma:</h3>
          <img 
            src={imageURL} 
            alt="Firma guardada" 
            style={styles.previewImage}
          />
        </div>
      )}
    </div>
  );
};

// Estilos
const styles = {
  container: {
    maxWidth: '600px',
    margin: '0 auto',
    padding: '20px',
    fontFamily: 'Arial, sans-serif',
    textAlign: 'center',
  },
  signatureContainer: {
    border: '1px solid #ccc',
    borderRadius: '4px',
    margin: '20px 0',
    backgroundColor: '#f9f9f9',
  },
  canvas: {
    backgroundColor: '#f9f9f9',
  },
  buttonGroup: {
    display: 'flex',
    justifyContent: 'center',
    gap: '10px',
    margin: '20px 0',
  },
  button: {
    padding: '10px 15px',
    backgroundColor: '#4CAF50',
    color: 'white',
    border: 'none',
    borderRadius: '4px',
    cursor: 'pointer',
    fontSize: '16px',
  },
  preview: {
    marginTop: '30px',
  },
  previewImage: {
    border: '1px solid #ddd',
    borderRadius: '4px',
    padding: '5px',
    maxWidth: '100%',
  },
  message: {
    color: '#666',
    margin: '10px 0',
  },
  success: {
    color: 'green',
    margin: '10px 0',
    fontWeight: 'bold',
  },
};

export default SignaturePad;