import React, { useState } from 'react';
import { Gantt } from 'gantt-task-react';
import "gantt-task-react/dist/index.css";
import './App.css';

// Recursos del proyecto
const recursos = [
  { id: "r1", nombre: "Juan Pérez" },
  { id: "r2", nombre: "María García" },
  { id: "r3", nombre: "Carlos López" },
  { id: "r4", nombre: "Ana Martínez" },
];

// Componente para el tooltip
const CustomTooltip = ({ task }) => {
  if (!task) return null;

  const formatFecha = (date) => date?.toLocaleDateString('es-ES') || 'No definida';

  const formatRecursos = (ids) => {
    if (!ids || !Array.isArray(ids)) return "Sin asignar";
    return ids.map(id => recursos.find(r => r.id === id)?.nombre).filter(Boolean).join(", ") || "Sin asignar";
  };

  return (
    <div className="custom-tooltip">
      <div><strong>{task.name}</strong></div>
      <div>Recursos: {formatRecursos(task.resources)}</div>
      <div>Fecha niciall: {formatFecha(task.start)}</div>
      <div>Fecha final: {formatFecha(task.end)}</div>
      <div>Duración: {task.duration || 0} días</div>
      <div>Avance: {task.progress || 0}%</div>
      {task.dependencies && (
        <div>Depende de: {task.dependencies.join(', ')}</div>
      )}
    </div>
  );
};

// Componente principal
const GanttEjemplo = () => {
  const [view, setView] = useState('Day');
  const [showTaskList, setShowTaskList] = useState(true);

  const [tasks, setTasks] = useState([
    {
      id: 'Proyecto',
      name: 'Proyecto de Desarrollo',
      start: new Date(2025, 4, 1),
      end: new Date(2025, 4, 25),
      progress: 45,
      type: 'project',
      hideChildren: false,
      styles: { progressColor: '#ff9e0d' },
      duration: 14,
    },
    {
      id: 'Analisis',
      name: 'Análisis de Requisitos',
      start: new Date(2025, 4, 1),
      end: new Date(2025, 4, 5),
      progress: 80,
      type: 'task',
      project: 'Proyecto',
      resources: [recursos[0].id, recursos[1].id],
      styles: { progressColor: '#4CAF50' },
      duration: 4,
    },
    {
      id: 'Diseno',
      name: 'Diseño de Arquitectura',
      start: new Date(2025, 4, 6),
      end: new Date(2025, 4, 10),
      progress: 30,
      type: 'task',
      project: 'Proyecto',
      dependencies: ['Analisis'], // Depende de Análisis
      resources: [recursos[2].id],
      styles: { progressColor: '#2196F3' },
      duration: 4,
    },
    {
      id: 'Implementacion',
      name: 'Implementación',
      start: new Date(2025, 4, 11),
      end: new Date(2025, 4, 20),
      progress: 10,
      type: 'task',
      project: 'Proyecto',
      dependencies: ['Diseno'], // Depende de Diseño
      resources: [recursos[1].id, recursos[3].id],
      styles: { progressColor: '#9C27B0' },
      duration: 9,
    },
    {
      id: 'Pruebas',
      name: 'Pruebas',
      start: new Date(2025, 4, 21),
      end: new Date(2025, 4, 25),
      progress: 0,
      type: 'task',
      project: 'Proyecto',
      dependencies: ['Implementacion'], // Depende de Implementación
      resources: [recursos[0].id, recursos[2].id],
      styles: { progressColor: '#F44336' },
      duration: 4,
    },
    {
      id: 'Cierre',
      name: 'Cierre del Proyecto',
      start: new Date(2025, 4, 30),
      end: new Date(2025, 4, 30),
      progress: 25,
      type: 'milestone',
      project: 'Proyecto',
      resources: [recursos[3].id],
      styles: { progressColor: '#607D8B' },
      duration: 29,
    },
  ]);

    // Colapsar/expandir todos los proyectos
    const toggleAllProjects = (collapse) => {
      setTasks(tasks.map(task => 
        task.type === 'project' 
          ? { ...task, hideChildren: collapse }
          : task
      ));
    };
    // Manejo de colapsar/expandir un proyecto
    const handleExpanderClick = (task) => {
      setTasks(tasks.map((t) => (t.id === task.id ? task : t)));
      console.log("On expander click Id:" + task.id);
    };

    // Componente personalizado para la cabecera de la lista de tareas
    const CustomTaskListHeader = () => {
      return (
        <div className="gantt-task-list-header" style={{ display: 'flex' }}>
          <div style={{ width: '100%', fontWeight: 'normal' }}>Nombre</div>
          {/*
          <div style={{ width: '33%', fontWeight: 'normal' }}>Inicio</div>
          <div style={{ width: '33%', fontWeight: 'normal' }}>Fin</div>
          */ }
        </div>
      );
    };

    // Componente de tabla de tareas personalizado
    const TaskListTable = ({ tasks }) => (
      <>
        {tasks.map((task) => (
          <div className="gantt-task-list-item" key={task.id} style={{ display: "flex", justifyContent: "flex-start", padding: "10px" }}>
            <div style={{ width: "100%", height: "22px", textWrap: "nowrap" ,padding: "8px 0px 0px 0px" }}>{task.name}</div>
          </div>
        ))}
      </>
    );
  return (
    <div className="gantt-container">
      <div className="gantt-controls">
        <button onClick={() => setView('Day')}>Día</button>
        <button onClick={() => setView('Week')}>Semana</button>
        <button onClick={() => setView('Month')}>Mes</button>
        <button onClick={() => setView('Year')}>Año</button>
        <label>
          <input
            type="checkbox"
            checked={showTaskList}
            onChange={() => setShowTaskList(!showTaskList)}
          />
          Mostrar nombres de tareas
        </label>

        <button onClick={() => toggleAllProjects(true)}>Colapsar Todos</button>
        <button onClick={() => toggleAllProjects(false)}>Expandir Todos</button>
      </div>

      <h2>Planificación del Proyecto</h2>
      
      <div className="gantt-chart-container">
        <Gantt
          tasks={tasks}
          onExpanderClick={handleExpanderClick}
          viewMode={view}
          locale="spa"
          listCellWidth={showTaskList ? true : false} // Si fijo poner "250px"
          TooltipContent={CustomTooltip}
          TaskListHeader={CustomTaskListHeader}
          TaskListTable={TaskListTable}
          columnWidth={70}
          todayColor="rgba(47, 0, 255, 0.1)"
        />
      </div>

      <div className="leyenda-recursos">
        <h3>Recursos asignados:</h3>
        <ul>
          {recursos.map((recurso, index) => (
            <li key={recurso.id}>
              <span className="color-recurso" style={{ 
                backgroundColor: `hsl(${index * 90}, 70%, 50%)` 
              }}></span>
              {recurso.nombre}
            </li>
          ))}
        </ul>
      </div>
    </div>
  );
};

export default GanttEjemplo;