import i18n from 'i18next';
import { initReactI18next } from 'react-i18next';
import LanguageDetector from 'i18next-browser-languagedetector';
import Backend from 'i18next-http-backend';

// Importar formatos adicionales
import { format, toDate } from 'date-fns';
import { enUS, es } from 'date-fns/locale';

// Importaciones como módulos ES
import enCommon from './en/common.json';
import esCommon from './es/common.json';

const dateFormats = {
  short: 'MM/dd/yyyy',
  long: 'MMMM dd, yyyy'
};

i18n
  .use(Backend)
  .use(LanguageDetector)
  .use(initReactI18next)
  .init({
    fallbackLng: 'en',
    debug: false,  // Habilita temporalmente para ver mensajes de consola
    interpolation: {
      escapeValue: false,
      format: (value, formatType, lng) => {
        if (value instanceof Date) {
          const locale = lng === 'es' ? es : enUS;
          return format(value, formatType, { locale });
        }
        
        if (formatType === 'currency') {
          return new Intl.NumberFormat(lng, {
            style: 'currency',
            useGrouping: true,
            minimumFractionDigits: 2,
            currency: lng === 'es' ? 'EUR' : 'USD'
          }).format(value);
        }
        
        return value;
      }
    },
    /*
    backend: {
      loadPath: '/i18n-react/locales/{{lng}}/{{ns}}.json',  // Ajustado manualmente
    },
    */
    // Para desarrollo, puedes cargar las traducciones directamente:
    resources: {
      en: {
        translation: {
          ...enCommon,
          dateFormats
        }  
      },
      es: {
        translation: {
          ...esCommon,
          dateFormats
        }
      }
    }
  });

export default i18n;