<?php
// src/MovieController.php

namespace App;

use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;
use PDO;
use PDOException;

class MovieController
{
    private $db;

    public function __construct(PDO $db)
    {
        $this->db = $db;
    }

    // --- Obtener todas las películas ---
    public function getAllMovies(Request $request, Response $response, array $args): Response
    {
        $stmt = $this->db->query("
            SELECT 
                m.id_movie, 
                m.name, 
                m.price, 
                m.startDate, 
                m.rating, 
                t.theme, 
                m.theme_id,
                s.support,
                m.support_id
            FROM movie m
            JOIN theme t ON m.theme_id = t.id_theme
            JOIN support s ON m.support_id = s.id_support
            ORDER BY m.id_movie ASC
        ");
        $movies = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $response->getBody()->write(json_encode($movies, JSON_UNESCAPED_UNICODE));
        return $response->withHeader('Content-Type', 'application/json');
    }

    // --- Obtener una película por ID ---
    public function getMovieById(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];
        $stmt = $this->db->prepare("
            SELECT 
                m.id_movie, 
                m.name, 
                m.price, 
                m.startDate, 
                m.rating, 
                t.theme, 
                m.theme_id,
                s.support,
                m.support_id
            FROM movie m
            JOIN theme t ON m.theme_id = t.id_theme
            JOIN support s ON m.support_id = s.id_support
            WHERE m.id_movie = :id
        ");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $movie = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$movie) {
            $response->getBody()->write(json_encode(['message' => 'Película no encontrada'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
        }

        $response->getBody()->write(json_encode($movie, JSON_UNESCAPED_UNICODE));
        return $response->withHeader('Content-Type', 'application/json');
    }

    // --- Crear una nueva película ---
    public function createMovie(Request $request, Response $response, array $args): Response
    {
        $data = $request->getParsedBody();

        // Validar datos de entrada (simple)
        if (!isset($data['name'], $data['price'], $data['startDate'], $data['rating'], $data['theme_id'], $data['support_id'])) {
            $response->getBody()->write(json_encode(['message' => 'Faltan campos obligatorios'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
        }

        $stmt = $this->db->prepare("
            INSERT INTO movie (name, price, startDate, rating, theme_id, support_id)
            VALUES (:name, :price, :startDate, :rating, :theme_id, :support_id)
        ");

        $stmt->bindParam(':name', $data['name']);
        $stmt->bindParam(':price', $data['price']);
        $stmt->bindParam(':startDate', $data['startDate']);
        $stmt->bindParam(':rating', $data['rating'], PDO::PARAM_INT);
        $stmt->bindParam(':theme_id', $data['theme_id'], PDO::PARAM_INT);
        $stmt->bindParam(':support_id', $data['support_id'], PDO::PARAM_INT);

        $stmt->execute();

        $newId = $this->db->lastInsertId();
        // Obtén la película completa recién creada si es necesario, o simplemente devuelve el nombre.
        // Para simplificar, devolvemos el nombre que se envió, y el nuevo ID.
        $response->getBody()->write(json_encode([
            'message' => 'Película creada',
            'id' => $newId,
            'name' => $data['name'] // <-- ¡AÑADIR ESTA LÍNEA!
        ], JSON_UNESCAPED_UNICODE));
        return $response->withHeader('Content-Type', 'application/json')->withStatus(201);
    }

    // --- Actualizar una película existente ---
    public function updateMovie(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];
        $data = $request->getParsedBody();

        // Validar datos de entrada (simple)
        if (!isset($data['name'], $data['price'], $data['startDate'], $data['rating'], $data['theme_id'], $data['support_id'])) {
            $response->getBody()->write(json_encode(['message' => 'Faltan campos obligatorios para la actualización'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
        }

        $stmt = $this->db->prepare("
            UPDATE movie
            SET name = :name, price = :price, startDate = :startDate, rating = :rating, theme_id = :theme_id, support_id = :support_id
            WHERE id_movie = :id
        ");

        $stmt->bindParam(':name', $data['name']);
        $stmt->bindParam(':price', $data['price']);
        $stmt->bindParam(':startDate', $data['startDate']);
        $stmt->bindParam(':rating', $data['rating'], PDO::PARAM_INT);
        $stmt->bindParam(':theme_id', $data['theme_id'], PDO::PARAM_INT);
        $stmt->bindParam(':support_id', $data['support_id'], PDO::PARAM_INT);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        $stmt->execute();

        if ($stmt->rowCount() === 0) {
            $response->getBody()->write(json_encode(['message' => 'Película no encontrada o sin cambios'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
        }

        $response->getBody()->write(json_encode([
            'message' => 'Película actualizada',
            'name' => $data['name'] // <-- ¡AÑADIR ESTA LÍNEA!
        ], JSON_UNESCAPED_UNICODE));
        return $response->withHeader('Content-Type', 'application/json');
    }

    // --- Eliminar una película ---
    public function deleteMovie(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];
        $stmt = $this->db->prepare("DELETE FROM movie WHERE id_movie = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() === 0) {
            $response->getBody()->write(json_encode(['message' => 'Película no encontrada'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
        }

        $response->getBody()->write(json_encode(['message' => 'Película eliminada'], JSON_UNESCAPED_UNICODE));
        return $response->withHeader('Content-Type', 'application/json');
    }
    // --- CRUD para Temas ---

    public function createTheme(Request $request, Response $response, array $args): Response
    {
        $data = $request->getParsedBody();

        if (!isset($data['theme']) || empty($data['theme'])) {
            $response->getBody()->write(json_encode(['message' => 'El nombre del tema es obligatorio.'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
        }

        try {
            $stmt = $this->db->prepare("INSERT INTO theme (theme) VALUES (:theme)");
            $stmt->bindParam(':theme', $data['theme']);
            $stmt->execute();

            $newId = $this->db->lastInsertId();
            $response->getBody()->write(json_encode([
                'message' => 'Tema creado con éxito.',
                'id' => $newId,
                'theme' => $data['theme']
            ], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(201);
        } catch (PDOException $e) {
            // Error si el tema ya existe (ej. columna 'theme' es UNIQUE)
            if ($e->getCode() === '23000') { // Código de error SQL para violación de unicidad
                $response->getBody()->write(json_encode(['message' => 'El tema ya existe.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(409); // 409 Conflict
            }
            $response->getBody()->write(json_encode(['message' => 'Error interno del servidor al crear el tema.', 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(500);
        }
    }

    public function updateTheme(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];
        $data = $request->getParsedBody();

        if (!isset($data['theme']) || empty($data['theme'])) {
            $response->getBody()->write(json_encode(['message' => 'El nombre del tema es obligatorio para la actualización.'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
        }

        try {
            $stmt = $this->db->prepare("UPDATE theme SET theme = :theme WHERE id_theme = :id");
            $stmt->bindParam(':theme', $data['theme']);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();

            if ($stmt->rowCount() === 0) {
                $response->getBody()->write(json_encode(['message' => 'Tema no encontrado o sin cambios.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
            }

            $response->getBody()->write(json_encode([
                'message' => 'Tema actualizado con éxito.',
                'id' => $id,
                'theme' => $data['theme']
            ], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json');
        } catch (PDOException $e) {
             if ($e->getCode() === '23000') {
                $response->getBody()->write(json_encode(['message' => 'El tema ya existe.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(409);
            }
            $response->getBody()->write(json_encode(['message' => 'Error interno del servidor al actualizar el tema.', 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(500);
        }
    }

    public function deleteTheme(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];

        // --- Manejo de Integridad Referencial (FASE POSTERIOR, AHORA SOLO ELIMINA) ---
        // Para la fase 5, aquí iría la verificación de si hay películas usando este tema.
        // Por ahora, simplemente intentamos eliminar. Si hay una restricción FOREIGN KEY,
        // la base de datos lanzará una excepción PDO.

        try {
            $stmt = $this->db->prepare("DELETE FROM theme WHERE id_theme = :id");
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();

            if ($stmt->rowCount() === 0) {
                $response->getBody()->write(json_encode(['message' => 'Tema no encontrado.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
            }

            $response->getBody()->write(json_encode(['message' => 'Tema eliminado con éxito.'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json');
        } catch (PDOException $e) {
            // Código de error 23000 es para violación de integridad (Foreign Key Constraint)
            if ($e->getCode() === '23000') {
                $response->getBody()->write(json_encode(['message' => 'No se puede eliminar el tema porque está asignado a una o más películas.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(409); // 409 Conflict
            }
            $response->getBody()->write(json_encode(['message' => 'Error interno del servidor al eliminar el tema.', 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(500);
        }
    }


    // --- CRUD para Soportes ---

    public function createSupport(Request $request, Response $response, array $args): Response
    {
        $data = $request->getParsedBody();

        if (!isset($data['support']) || empty($data['support'])) {
            $response->getBody()->write(json_encode(['message' => 'El nombre del soporte es obligatorio.'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
        }

        try {
            $stmt = $this->db->prepare("INSERT INTO support (support) VALUES (:support)");
            $stmt->bindParam(':support', $data['support']);
            $stmt->execute();

            $newId = $this->db->lastInsertId();
            $response->getBody()->write(json_encode([
                'message' => 'Soporte creado con éxito.',
                'id' => $newId,
                'support' => $data['support']
            ], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(201);
        } catch (PDOException $e) {
            if ($e->getCode() === '23000') {
                $response->getBody()->write(json_encode(['message' => 'El soporte ya existe.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(409);
            }
            $response->getBody()->write(json_encode(['message' => 'Error interno del servidor al crear el soporte.', 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(500);
        }
    }

    public function updateSupport(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];
        $data = $request->getParsedBody();

        if (!isset($data['support']) || empty($data['support'])) {
            $response->getBody()->write(json_encode(['message' => 'El nombre del soporte es obligatorio para la actualización.'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
        }

        try {
            $stmt = $this->db->prepare("UPDATE support SET support = :support WHERE id_support = :id");
            $stmt->bindParam(':support', $data['support']);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();

            if ($stmt->rowCount() === 0) {
                $response->getBody()->write(json_encode(['message' => 'Soporte no encontrado o sin cambios.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
            }

            $response->getBody()->write(json_encode([
                'message' => 'Soporte actualizado con éxito.',
                'id' => $id,
                'support' => $data['support']
            ], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json');
        } catch (PDOException $e) {
            if ($e->getCode() === '23000') {
                $response->getBody()->write(json_encode(['message' => 'El soporte ya existe.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(409);
            }
            $response->getBody()->write(json_encode(['message' => 'Error interno del servidor al actualizar el soporte.', 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(500);
        }
    }

    public function deleteSupport(Request $request, Response $response, array $args): Response
    {
        $id = $args['id'];

        // --- Manejo de Integridad Referencial (FASE POSTERIOR, AHORA SOLO ELIMINA) ---
        // Para la fase 5, aquí iría la verificación de si hay películas usando este soporte.
        // Por ahora, simplemente intentamos eliminar.

        try {
            $stmt = $this->db->prepare("DELETE FROM support WHERE id_support = :id");
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();

            if ($stmt->rowCount() === 0) {
                $response->getBody()->write(json_encode(['message' => 'Soporte no encontrado.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(404);
            }

            $response->getBody()->write(json_encode(['message' => 'Soporte eliminado con éxito.'], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json');
        } catch (PDOException $e) {
            if ($e->getCode() === '23000') {
                $response->getBody()->write(json_encode(['message' => 'No se puede eliminar el soporte porque está asignado a una o más películas.'], JSON_UNESCAPED_UNICODE));
                return $response->withHeader('Content-Type', 'application/json')->withStatus(409);
            }
            $response->getBody()->write(json_encode(['message' => 'Error interno del servidor al eliminar el soporte.', 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE));
            return $response->withHeader('Content-Type', 'application/json')->withStatus(500);
        }
    }
}