# Introduction

zend-memory assists with managing data in an environment with limited memory.

Memory objects (memory containers) are generated by the memory manager at your
request and transparently swapped/loaded when necessary.

For example, if creating or loading a managed object would cause the total
memory usage to exceed the limit you specify, some managed objects are copied to
cache storage outside of memory. In this way, the total memory used by managed
objects does not exceed the limit you need to enforce. To provide this
functionality, the memory manager can compose [zend-cache storage adapters](http://zendframework.github.io/zend-cache/storage/adapter/)
as storage providers.

## Usage

Instantiate the memory manager class:

```php
use Zend\Memory\MemoryManager;

// No caching backend:
$memoryManager = new MemoryManager();
```

Optionally, you can create a cache storage adapter, and pass it to the
`MemoryManager` constructor:

```php
use Zend\Cache\StorageFactory;
use Zend\Memory\MemoryManager;

// Use a filesystem adapter, placing memory blocks in the tmp directory
// under which the application is running.
$cache = StorageFactory::factory([
    'adapter' => [
        'name' => 'filesystem',
        'options' => ['cache_dir' => './tmp/'],
    ],
]);

$memoryManager = new MemoryManager($cache);
```

Once you have a `MemoryManager` instance, you can start pushing values to it and
pulling values from it.

```php
$loadedFiles = array();

for ($count = 0; $count < 10000; $count++) {
    $f = fopen($fileNames[$count], 'rb');
    $data = fread($f, filesize($fileNames[$count]));
    $fclose($f);

    $loadedFiles[] = $memoryManager->create($data);
}

echo $loadedFiles[$index1]->value;

$loadedFiles[$index2]->value = $newValue;

$loadedFiles[$index3]->value[$charIndex] = '_';
```

## Theory of Operation

zend-memory operates with the following concepts:

- Memory manager
- Memory container
  - Locked memory object
  - Movable memory object

### Memory manager

The memory manager generates memory objects (locked or movable) by request of
the application, and returns them wrapped into a memory container object.

### Memory container

The memory container has a virtual or actual `value` attribute of type `string`.
This attribute contains the data value specified at memory object creation time.

You can operate with this `value` attribute as an object property:

```php
$memObject = $memoryManager->create($data);

echo $memObject->value;

$memObject->value = $newValue;

$memObject->value[$index] = '_';

echo ord($memObject->value[$index1]);

$memObject->value = substr($memObject->value, $start, $length);
```

### Locked memory

Locked memory objects are always stored in memory. Data stored in locked memory
are never swapped to the cache backend.

### Movable memory

Movable memory objects are transparently swapped and loaded to/from the cache
backend by the `MemoryManager` when necessary.

The memory manager does not swap objects with size less than the specified
minimum, due to performance considerations. See [the settings section](memory-manager.md#settings)
for more details.
